<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\EmployeeInformation;
use App\Models\CpParmCodes;
use App\Http\Requests\UserStoreRequest;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;
use Illuminate\Mail\Message;
use Database\Seeders\LeaveTypesSeeder;
use Database\Seeders\AnnualLeaveEntitlementSeeder;
use Database\Seeders\LeaveBalancesSeeder;
use Database\Seeders\LeaveGroupsSeeder;
use Database\Seeders\PasswordComplexitySeeder;
use Database\Seeders\PublicHolidaysSeeder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Models\PasswordComplexity;


class UserController extends Controller
{
    public function getCsrfToken()
    {
        $csrfToken = csrf_token();
        return response()->json(['csrfToken' => $csrfToken]);
    }

    public function index(){
        $users = User::all();

        return response()->json([
            'results' => $users
        ], 200);
    }

    public function register(Request $request)
    {
        $seeder = new LeaveTypesSeeder();
        $seeder->run();
        $seeder = new LeaveGroupsSeeder();
        $seeder->run();
        $seeder = new PasswordComplexitySeeder();
        $seeder->run();
        $seeder = new PublicHolidaysSeeder();
        $seeder->run();

        // return Hash::make($request->password);
        try {
            DB::beginTransaction();

            $emails = User::pluck('email');
            // $valueExists = YourModel::whereIn('column_name', ['value1', 'value2', 'value3'])->exists();
            if ($emails->contains($request->email)) {
                return response()->json([
                    'message' => "User already exists."
                ],417);
            }

            // Create User
            User::create([
                'employee_number' => $request->employee_number,
                'surname' => $request->surname,
                'name' => $request->name,
                'initials' => $request->initials,
                'email' => $request->email,
                'password' => $request->password,
                'leave_group_id' => $request->gender === "F" ? 1 : 2,
                'access_level' => "2",
            ]);
            
            EmployeeInformation::create([
                'employee_number' => $request->employee_number,
                "gender" => $request->gender,
                "payroll_number" => $request->payroll_number,
                "cost_codes" => $request->cost_codes,
                "occupation" => $request->occupation,
                "pay_point" => $request->pay_point,
                "title" => $request->title,
                "birth_date" => $request->birth_date,
                "engage_date" => $request->engage_date,
                "discharge_date" => $request->discharge_date,
                "stop_accrual" => 0,
            ]);

            $seeder = new LeaveBalancesSeeder($request);
            $seeder->run();
            $seeder = new AnnualLeaveEntitlementSeeder($request);
            $seeder->run();

            DB::commit();
 
            // Return Json Response
            return response()->json([
                'message' => "User successfully created."
            ],200);
        } catch (\Exception $e) {
            DB::rollBack();
            // Return Json Response
            return response()->json([
                'message' => "Something went really wrong!" .$e
            ],500);
        }
    }

    public function show($id){
        $users = User::find($id);

        if(!$users){
            return response()->json([
                'message' => "User not found"
            ], 404);
        }

        return response()->json([
            'results' => $users
        ], 200);
    }

    public function update(UserStoreRequest $request, $id)
    {
        try {
            // Find User
            $users = User::find($id);
            if(!$users){
              return response()->json([
                'message'=>'User Not Found.'
              ],404);
            }
       
            //echo "request : $request->image";
            $users->name = $request->name;
            $users->email = $request->email;
       
            // Update User
            $users->save();
       
            // Return Json Response
            return response()->json([
                'message' => "User successfully updated."
            ],200);
        } catch (\Exception $e) {
            // Return Json Response
            return response()->json([
                'message' => "Something went really wrong!"
            ],500);
        }
    }

    public function destroy($id)
    {
        // Detail 
        $users = User::find($id);
        if(!$users){
          return response()->json([
             'message'=>'User Not Found.'
          ],404);
        }
         
        // Delete User
        $users->delete();
       
        // Return Json Response
        return response()->json([
            'message' => "User successfully deleted."
        ],200);
    }

    public function login(Request $request)
    {
        $user= User::where('email', $request->email)->first();

        if ($user && $user->password === null) {
            return response([
                'message' => ['Please set up your password first before using the system.'],
                'email' => $request->email,
                'action' => 'password'
            ], 404);
        }

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response([
                'message' => ['These credentials do not match our records.'],
                'action' => 'failed'
            ], 404);
        }
    
        $token = $user->createToken('my-app-token')->plainTextToken;
    
        $response = [
            'employee' => $user,
            'token' => $token
        ];
    
        return response($response, 200);
    }

    public function loginWithToken(Request $request)
    {
        $token = $request->input('token');
        $id_token = explode('|', $token);
        $id = $id_token[0];
        $token = $id_token[1];

        // Verify the token
        $personalAccessToken = DB::table('personal_access_tokens')
            ->where('id', $id)
            ->where('token', hash('sha256', $token))
            ->first();

        if ($personalAccessToken) {
            $expiresAtDate = Carbon::parse($personalAccessToken->expires_at);
            $currentDate = Carbon::now();

            if ($currentDate->gt($expiresAtDate) && $personalAccessToken->expires_at) {
                return response()->json([
                    'message' => "Token Expired."
                ],401);
            }
            // Log in the user
            $user = User::find($personalAccessToken->tokenable_id);

            $response = [
                'employee' => $user,
                'token' => $token
            ];
        
            return response($response, 200);
        }

        return response()->json(['error' => 'Invalid token'], 401);
    }

    public function logout(Request $request)
    {
        Auth::user()->currentAccessToken()->delete();

        return response()->json(['message' => 'Successfully logged out']);
    }

    public function getEmployees(){
        $employees = User::all();

        return response()->json([
            'results' => $employees
        ], 200);
    }

    public function batchImport(Request $request)
    {
        $user= User::where('email', $request->super_admin_email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response([
                'message' => ['These credentials do not match our records.'],
                'success' => false,
            ], 404);
        }

        try {
            DB::beginTransaction();

            if ($request->has('employees')) {
                foreach ($request->employees as $employee){
                    $employee = (object) $employee;

                    $emails = User::pluck('email');
                    // $valueExists = YourModel::whereIn('column_name', ['value1', 'value2', 'value3'])->exists();
                    if ($emails->contains($employee->email) && $employee->email !== "") {
                        return response()->json([
                            'success' => false,
                            'message' => "User already exists."
                        ],417);
                    }

                    $existing_empNo = User::where('employee_number', $employee->employee_number)->first();
                    if ($existing_empNo) {
                        $id = $existing_empNo->id;
                    } else {
                        $id = null;
                    }
                    
                    User::updateOrCreate(
                        ['id' => $id ],
                        [
                            'employee_number' => $employee->employee_number,
                            'surname' => $employee->surname,
                            'name' => $employee->name,
                            'initials' => $employee->initials,
                            'email' => $employee->email,
                            'password' => $employee->password,
                            'leave_group_id' => $employee->gender === "F" ? 1 : 2,
                            'access_level' => "3"
                        ],
                    );
                    
                    $existing_empNo = EmployeeInformation::where('employee_number', $employee->employee_number)->first();
                    if ($existing_empNo) {
                        $id = $existing_empNo->id;
                    } else {
                        $id = null;
                    }

                    EmployeeInformation::updateOrCreate(
                        ['id' => $id ],
                        [
                            "employee_number" => $employee->employee_number,
                            "gender" => $employee->gender,
                            "payroll_number" => $employee->payroll_number,
                            "cost_codes" => $employee->cost_codes,
                            "occupation" => $employee->occupation,
                            "pay_point" => $employee->pay_point,
                            "title" => $employee->title,
                            "birth_date" => $employee->birth_date,
                            "engage_date" => $employee->engage_date,
                            "discharge_date" => $employee->discharge_date,
                            "stop_accrual" => 0
                        ]
                    );

                    // EmployeeInformation::create([
                    //     'employee_number' => $employee->employee_number,
                    //     "gender" => $employee->gender,
                    //     "payroll_number" => $employee->payroll_number,
                    //     "cost_codes" => $employee->cost_codes,
                    //     "occupation" => $employee->occupation,
                    //     "pay_point" => $employee->pay_point,
                    //     "title" => $employee->title,
                    //     "birth_date" => $employee->birth_date,
                    //     "engage_date" => $employee->engage_date,
                    //     "discharge_date" => $employee->discharge_date,
                    //     "stop_accrual" => 0,
                    // ]);


                    // // Create User
                    // User::create([
                    //     'employee_number' => $employee->employee_number,
                    //     'surname' => $employee->surname,
                    //     'name' => $employee->name,
                    //     'initials' => $employee->initials,
                    //     'email' => $employee->email,
                    //     'password' => $employee->password,
                    //     'leave_group_id' => $employee->gender === "F" ? 1 : 2,
                    //     'access_level' => "3",
                    // ]);
                    

                    $seeder = new LeaveBalancesSeeder($employee);
                    $seeder->run();
                    $seeder = new AnnualLeaveEntitlementSeeder($employee);
                    $seeder->run();
        
                }   
            }
            
            if ($request->has('cp_parm_codes')) {
                $rows = $request->cp_parm_codes;
                // $req = (object) $req;

                foreach ($rows as $req){
                    //update or create
                    $employee = CpParmCodes::updateOrCreate(
                        ['OrdinalNo' => $req["OrdinalNo"]],  // Search criteria
                        [
                            "AdjustFlag" => is_array($req["AdjustFlag"] ?? "") ? json_encode($req["AdjustFlag"] ?? "") : $req["AdjustFlag"] ?? "",
                            "CodeName" => is_array($req["CodeName"] ?? "") ? json_encode($req["CodeName"] ?? "") : $req["CodeName"] ?? "",
                            "CodeType" => is_array($req["CodeType"] ?? "") ? json_encode($req["CodeType"] ?? "") : $req["CodeType"] ?? "",
                            "ConsolArnFlag" => is_array($req["ConsolArnFlag"] ?? "") ? json_encode($req["ConsolArnFlag"] ?? "") : $req["ConsolArnFlag"] ?? "",
                            "ConsolCode" => is_array($req["ConsolCode"] ?? "") ? json_encode($req["ConsolCode"] ?? "") : $req["ConsolCode"] ?? "",
                            "Limit" => is_array($req["Limit"] ?? "") ? json_encode($req["Limit"] ?? "") : $req["Limit"] ?? "",
                            "ManualDisplayInd" => is_array($req["ManualDisplayInd"] ?? "") ? json_encode($req["ManualDisplayInd"] ?? "") : $req["ManualDisplayInd"] ?? "",
                            "OrdinalNo" => is_array($req["OrdinalNo"] ?? "") ? json_encode($req["OrdinalNo"] ?? "") : $req["OrdinalNo"] ?? "",
                            "Payroll" => is_array($req["Payroll"] ?? "") ? json_encode($req["Payroll"] ?? "") : $req["Payroll"] ?? "",
                            "ProRataInd" => is_array($req["ProRataInd"] ?? "") ? json_encode($req["ProRataInd"] ?? "") : $req["ProRataInd"] ?? "",
                            "QmfDisplayInd" => is_array($req["QmfDisplayInd"] ?? "") ? json_encode($req["QmfDisplayInd"] ?? "") : $req["QmfDisplayInd"] ?? "",
                            "TaxIncAsn" => is_array($req["TaxIncAsn"] ?? "") ? json_encode($req["TaxIncAsn"] ?? "") : $req["TaxIncAsn"] ?? "",
                            "TaxType" => is_array($req["TaxType"] ?? "") ? json_encode($req["TaxType"] ?? "") : $req["TaxType"] ?? "",
                            "currency" => is_array($req["currency"] ?? "") ? json_encode($req["currency"] ?? "") : $req["currency"] ?? "",
                            "TamsColumn" => in_array("TamsColumn", $req) ? (is_array($req["TamsColumn"] ?? "") ? json_encode($req["TamsColumn"] ?? "") : $req["TamsColumn"] ?? "") : "",
                            "Journal" => in_array("Journal", $req) ? (is_array($req["Journal"] ?? "") ? json_encode($req["Journal"] ?? "") : $req["Journal"] ?? "") : "",
                        ]
                    );
                }

                DB::commit();

                $parmCodes = CpParmCodes::all();

                // $json_array['cp_parm_codes'] = $parmCodes;
            }  

            DB::commit();

            // Return Json Response
            return response()->json([
                'success' => true,
                'message' => "Users successfully uploaded."
            ],200);
        } catch (\Exception $e) {
            // get first line only
            $error = explode("\n", $e->getMessage())[0];
            DB::rollBack();

            // Return Json Response
            return response()->json([
                'success' => false,
                'message' => "Something went really wrong! . $error"
            ],500);
        }
    }

    public function resetPassword(Request $request)
    {
        try{
            $this->validate($request, [
                'token' => 'required',
                'email' => 'required|email',
                'password' => 'required|confirmed',
                // 'password' => 'required|confirmed|min:8',
            ]);

            $settings = PasswordComplexity::find(1);

            $rules = [
                'password' => [
                    'required',
                    'string',
                    'min:' . $settings->min_length,
                ],
            ];

            $message = [];

            if ($settings->include_numbers) {
                $rules['password'][] = 'regex:/[0-9]/';
                if (!preg_match('/[0-9]/', $request->all()["password"])) {
                    $message[] = 'Password must contain at least one number.';
                }
            }

            if ($settings->include_special_chars) {
                $rules['password'][] = 'regex:/[\W_]/';
                if (!preg_match('/[\W_]/', $request->all()["password"])) {
                    $message[] = 'Password must contain at least one special character.';
                }
            }

            if ($settings->include_uppercase) {
                $rules['password'][] = 'regex:/[A-Z]/';
                if (!preg_match('/[A-Z]/', $request->all()["password"])) {
                    $message[] = 'Password must contain at least one uppercase letter.';
                }
            }


            $validator = Validator::make($request->all(), [
                'password' => $rules['password']
            ]);

            if ($validator->fails()) {
                return response()->json([
                    // "message" => $validator->errors()->first()
                    "message" => ($validator->errors()->first() == "The password field format is invalid." 
                            	? implode("\n", $message)
                                : $validator->errors()->first())
                ], 422);
            }

        } catch (\Exception $exception) {
            //if passwords don't match return error
            if($request->password !== $request->password_confirmation){
                return response()->json([
                    'success' => false,
                    'message' => "Passwords don't match."
                ], 417);
            }

            $response = [
                'success' => false,
                'message' => 'Fill in all fields '.$exception
            ];
            return response($response, 417);
        }

        // if(!$user->email_verified_at()){
        //     return response()->json([
        //         'message' => "Initial Registration to be completed here or !user->email_verified_at to point to this function"
        //         // 'message' => "User not verified."
        //     ], 404);
        // }

        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function ($user, $password) {
                $userData = [
                    'password' => Hash::make($password),
                    'remember_token' => Str::random(60),
                ];
                
                if ($user->email_verified_at === null) {
                    $userData['email_verified_at'] = Carbon::now();
                }
                
                $user->forceFill($userData)->save();
                // Perform any additional actions after password reset if needed
            }
        );

        // Return appropriate response based on status
        if($status === Password::PASSWORD_RESET){
            return response()->json([
                'status' => 'success',
                'message' => __($status),
            ], 200);
        } else {
            return response()->json([
                'status' => 'error',
                'message' => __($status),
            ], 417);
        }
    }

    public function forgotPassword(Request $request)
    {
        try{
            $this->validate($request, [
                'email' => 'required|email',
            ]);
        } catch (\Exception $exception) {
            $response = [
                'success' => false,
                'message' => 'missing post field'
            ];
            return response($response, 417);
        }

        $status = Password::sendResetLink(
            $request->only('email'),
        );

        // Return appropriate response based on status
        if($status === Password::RESET_LINK_SENT){
            return response()->json([
                'status' => 'success',
                'message' => __($status),
            ], 200);
        } else {
            return response()->json([
                'status' => 'error',
                'message' => __($status),
            ], 417);
        }
    }

}
