<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Models\LeaveType;
use App\Models\LeaveGroup;
use App\Models\LeaveBalance;
use App\Models\LeaveTaken;
use App\Models\User;
use App\Models\AccrualRun;
use App\Models\AnnualLeaveEntitlement;
use App\Models\PublicHoliday;
use App\Models\LinkedAccount;
use App\Models\PayrollSetting;
use App\Models\CpParmCodes;
use App\Models\EmployeeInformation;
use App\Mail\LeaveApprovalRequestNotification;
use App\Mail\LeaveActionedNotification;
use Carbon\Carbon;
use DateTime;

class LeaveController extends Controller
{
    public function getLeavePlannerData($id){
        try {   
            $employee_leave_taken = $this->getEmployeeLeaveTaken($id);

            $leave_types = LeaveType::all();

            $leave_balances = $this->leaveBalancesToJson("all");
            // $approvee_leave_taken = $this->getApproveeLeaveTaken($id);
            $leave_groups = $this->modifyLeaveGroups();
            $accrual_runs = AccrualRun::all();
            //annual_leave_entitlement where employee_number != 0
            $annual_leave_entitlement = AnnualLeaveEntitlement::where('employee_number', '!=', 1000)->get();
            // $annual_leave_entitlement = AnnualLeaveEntitlement::all();
            $public_holidays = PublicHoliday::all();
            $linked_accounts = LinkedAccount::all();
            $payroll_settings = PayrollSetting::all();
            $cp_parm_codes = CpParmCodes::all();

            foreach ($public_holidays as $holiday) {
                #check if date falls on a sunday and add a day to make it monday
                $holiday->date = Carbon::createFromFormat('Y-m-d', $holiday->date);
                
                if ($holiday->date->isSunday() && $holiday->name != "Easter Sunday") {
                    $holiday->date->addDay();
                }
                
                $holiday->date = $holiday->date->format('Y-m-d');
            };

            // $employees = User::all();
            // $employee_information = EmployeeInformation::all();

            // //merge employees and employee_information based on employee_number
            // foreach ($employees as $employee) {
            //     foreach ($employee_information as $employee_info) {
            //         if ($employee->employee_number == $employee_info->employee_number) {
            //             $employee->employee_information = $employee_info;
            //         }
            //     }
            // }

            $employees = User::where('employee_number', '!=', 1000)->with('employeeInformation')->get();
            // $employees = User::with('employeeInformation')->get();

            // Return Json Response
            return response()->json([
                'success' => true,
                'leave_types' => $leave_types,
                'leave_groups' => $leave_groups,
                'leave_taken' => $employee_leave_taken->original["results"],
                'leave_balances' => $leave_balances,
                'accrual_runs' => $accrual_runs,
                'employees' => $employees,
                'public_holidays' => $public_holidays,
                'annual_leave_entitlement' => $annual_leave_entitlement,
                'linked_accounts' => $linked_accounts,
                'payroll_settings' => $payroll_settings,
                'cp_parm_codes' => $cp_parm_codes,
                // 'approvee_leave_taken' => $approvee_leave_taken->original["results"],
            ],200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ],500);
        }
    }

    public function editEmployee(Request $request){
        $rows = $request->rows;
        // return response()->json([
        //     'message' => "Leave failed to edit.",
        //     'request' => $rows
        // ],417);
        DB::beginTransaction();
        
        try {
            // & is a reference to the original source so changes are made to original source rather than copy
            foreach ($rows as &$req) {
                // $employee = User::findOrFail($req["id"]);
                $employee = User::where('employee_number', $req["employee_number"])->first();

                $employee->leave_group_id = $req["leave_group_id"];
                $employee->access_level = $req["access_level"];
                $employee->save();

                $employee_information = EmployeeInformation::where('employee_number', $req["employee_number"])->first();
                $employee_information->stop_accrual = $req["employee_information"]["stop_accrual"];
                $employee_information->save();
            }

            DB::commit();

            $employees = User::where('employee_number', '!=', 1000)->with('employeeInformation')->get();

            return response()->json([
                'message' => "Employee successully edited.",
                'data' => $employees
            ],200); 
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }     

    public function getAllLeaveTaken(){
        $leave_taken = LeaveTaken::all();

        return response()->json([
            'results' => $leave_taken
        ], 200);
    }

    public function actionLeaveTaken(Request $request){
        $rows = $request->rows;

        DB::beginTransaction();
        
        try {
            // & is a reference to the original source so changes are made to original source rather than copy
            foreach ($rows as &$req) {
                $req_array = $req;
                $req = (object) $req;
                
                // $leave_taken = LeaveTaken::findOrFail($req->id);
                $leave_taken = LeaveTaken::find($req->id);
                $employee = User::where('employee_number', $leave_taken->employee_number)->first();
                $email_response = "";

                //if any variable from frontend differs from that in DB or backend or leave_taken is empty
                $leave_taken_array = $leave_taken->toArray();
                
                Arr::forget($leave_taken_array, ['actioned_by', 'action_reason', 'created_at', 'updated_at', 'status']);
                Arr::forget($req_array, ['actioned_by', 'action_reason', 'created_at', 'updated_at', 'status']);

                $req_array["start_date"] = (count(explode('T', $leave_taken_array["start_date"])) > 1 
                                            ? $req_array["start_date"] 
                                            : Carbon::parse($req_array["start_date"])->format('Y-m-d'));
                $req_array["end_date"] = (count(explode('T', $leave_taken_array["end_date"])) > 1 
                                            ? $req_array["end_date"] 
                                            : Carbon::parse($req_array["end_date"])->format('Y-m-d'));

                $differences = array_diff_assoc($leave_taken_array, $req_array);

                if(count($differences) > 0 || $leave_taken->count() == 0) {
                    $approvee_leave_taken = $this->getApproveeLeaveTaken($request->editor_employee_number);
                    $employee_leave_taken = $this->getEmployeeLeaveTaken($request->editor_employee_number);

                    return response()->json([
                        'message' => $leave_taken->count() > 0 ? "Leave details differs. Employee might have edited this leave. Please review the new leave" : "Employee deleted leave.",
                        'approvee_leave_taken' => $approvee_leave_taken->original["results"],
                        'employee_leave_taken' => $employee_leave_taken->original["results"],
                        'email_response' => $email_response
                    ],202); //205 Accepted
                }


                //if leave taken status is not already the same as req status
                if($leave_taken->status !== $req->status) {
                    // $leave_group = LeaveGroup::where('id', $employee->leave_group_id)->get()[0];
                    $leave_type = LeaveType::where('id', $leave_taken->leave_type_id)->get()[0];
                    $approvers = json_decode($leave_type->leave_approvers);
                    //check if length of array approvers is greater than 1 and approver is not the last to approve and type is heirachichal and action is not to decline
                    if((count($approvers) > 1) && (end($approvers) !== $request->editor_employee_number) && ($leave_type->leave_approvers_type === "hierachichal") && ($req->status !== "declined")) {
                        // if($leave_taken->status === "pending"){
                        if ($leave_taken->actioned_by === null){
                            $leave_taken->actioned_by = json_encode([$request->editor_employee_number]);
                            $leave_taken->action_reason = json_encode([$req->action_reason]);
                        } else {
                            $list1 = json_decode($leave_taken->actioned_by);
                            $list2 = json_decode($leave_taken->action_reason);
                            $list1[] = $request->editor_employee_number;
                            $list2[] = $req->action_reason;
                            $leave_taken->actioned_by = json_encode($list1);
                            $leave_taken->action_reason = json_encode($list2);
                        }
                        
                        $leave_taken->save();

                        DB::commit();

                        $current_editor_index = array_search($request->editor_employee_number, $approvers);
    
                        // $succeeding_editor_employee_number = User::findOrFail($approvers[$current_editor_index+1]);
                        $succeeding_editor_employee_number = User::where('employee_number', $approvers[$current_editor_index+1])->first();

                        $email_data = [
                            "title" => "Leave Approval Request",
                            "email" => $succeeding_editor_employee_number->email,
                            "blade" => 'emails.leave_approval_request',
                            "displayableActionUrl" => env("FRONTEND_URL"),
                            "actionUrl" => env("FRONTEND_URL"),
                            "color" => "blue",
                            "greeting" => "Good day",
                            "introLines" => ["You have a new leave request from ".$employee->name." ".$employee->surname],
                            "actionText" => "Login",
                            "outroLines" => ["Please review the request and action."],
                            "salutation" => "Regards,\n".env("APP_NAME") 
                        ];
                        $email_data = (object) $email_data;
                        $email_response = $this->sendLeaveRequestEmail($email_data, "leave approval request");
                    } else {
                        $leave_taken->status = $req->status;
                        //checking if actioned by is null
                        if($leave_taken->actioned_by === null){
                            $leave_taken->actioned_by = json_encode([$request->editor_employee_number]);
                            $leave_taken->action_reason = json_encode([$req->action_reason]);
                        } else {
                            $list1 = json_decode($leave_taken->actioned_by);
                            $list2 = json_decode($leave_taken->action_reason);
                            $list1[] = $request->editor_employee_number;
                            $list2[] = $req->action_reason;
                            $leave_taken->actioned_by = json_encode($list1);
                            $leave_taken->action_reason = json_encode($list2);
                        }
                        $leave_taken->save();
    
                        //FIX LEAVE BALANCE TO REMOVE PENDING
                        $modify_leave_balances = $this->modifyLeaveBalances($leave_taken, $req, $req->status);
    
                        if($modify_leave_balances[1] === 417) {
                            DB::rollBack();
    
                            return response()->json([
                                'message' => $modify_leave_balances[0]
                            ], 417);
                        }
                        /////
                        DB::commit();
    
                        // $employee = User::findOrFail($leave_taken->employee_number);
                        // $editor = User::findOrFail($request->editor_employee_number);
                        $employee = User::where('employee_number', $leave_taken->employee_number)->first();
                        $editor = User::where('employee_number', $request->editor_employee_number)->first();
    
                        $email_data = [
                            "title" => "Leave Approval Request",
                            "email" => $employee->email,
                            "blade" => 'emails.leave_actioned',
                            "displayableActionUrl" => env("FRONTEND_URL")."/calendar",
                            "actionUrl" => env("FRONTEND_URL")."/calendar",
                            "color" => "blue",
                            "greeting" => "Good day",
                            "introLines" => ($req->status !== "pending" 
                                            ? (($req->status === "declined") 
                                                ? ["Your leave request from ".$leave_taken->start_date." to ".$leave_taken->end_date." has been ".$req->status,
                                                    "by ".$editor->name." ".$editor->surname] 
                                                : ["Your leave request from ".$leave_taken->start_date." to ".$leave_taken->end_date." has been ".$req->status])
                                            : ["Your leave days from ".$leave_taken->start_date." to ".$leave_taken->end_date." have been restored"]),
                            "actionText" => "Leave Planner",
                            "outroLines" => ["You can click the button above to view your leave planner."],
                            "salutation" => "Regards,\n".env("APP_NAME") 
                        ];
                        $email_data = (object) $email_data;
                        $email_response = $this->sendLeaveRequestEmail($email_data, "leave actioned");
                    }

                }
            }

            $approvee_leave_taken = $this->getApproveeLeaveTaken($request->editor_employee_number);
            $employee_leave_taken = $this->getEmployeeLeaveTaken($request->editor_employee_number);

            return response()->json([
                'message' => "Leave successully actioned.",
                'approvee_leave_taken' => $approvee_leave_taken->original["results"],
                'employee_leave_taken' => $employee_leave_taken->original["results"],
                'email_response' => $email_response
            ],200); 

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }     

    public function addLeaveType(Request $request){
        $rows = $request->rows;
        
        DB::beginTransaction();

        try {
            foreach ($rows as &$req) {
                LeaveType::create([
                    'name' => $req["name"],
                    'accrual_type' => $req["accrual_type"],
                    'accrual_days' => $req["accrual_days"],
                    'start_time' => $req["start_time"],
                    'end_time' => $req["end_time"],
                    "monday" => $req["monday"],
                    "tuesday" => $req["tuesday"],
                    "wednesday" => $req["wednesday"],
                    "thursday" => $req["thursday"],
                    "friday" => $req["friday"],
                    "saturday" => $req["saturday"],
                    "sunday" => $req["sunday"]
                ]);
            }

            DB::commit();

            $leave_types = LeaveType::all();
 
            // Return Json Response
            return response()->json([
                'message' => "Leave Type successfully added.",
                'data' => $leave_types
            ],200);
        } catch (\Exception $e) {
            // Return Json Response
            return response()->json([
                'message' => "Something went really wrong! . $e"
            ],500);
        }
    }

    public function getLeaveTypes(){
        $leave_types = LeaveType::all();

        // json decode the leave_approvers field for all records
        foreach ($leave_types as &$leave_type) {
            // $leave_type->leave_approvers = [];
            $leave_type["leave_approvers"] = json_decode($leave_type["leave_approvers"]);
        }

        return response()->json([
            'results' => $leave_types
        ], 200);
    }

    public function editLeaveType(Request $request){
        // return $request;
        $rows = $request->rows;
        // return response()->json([
        //     'message' => "Leave failed to edit.",
        //     'request' => $rows
        // ],417);
        DB::beginTransaction();
        
        try {
            // & is a reference to the original source so changes are made to original source rather than copy

            foreach ($rows as &$req) {
                // $leave_type = LeaveType::findOrFail($req["id"]);

                //updateOrCreate
                $leave_type = LeaveType::updateOrCreate(    
                    ['id' => $req['id']],        
                    [
                        'name' => ($req["id"] !== 1) ? $req["name"] : "ANNUAL",
                        'accrual_type' => $req["accrual_type"],
                        'accrual_days' => $req["accrual_days"],
                        'start_time' => $req["start_time"],
                        'end_time' => $req["end_time"],
                        'leave_approvers' => $req["leave_approvers"],
                        'leave_approvers_type' => $req["leave_approvers_type"],
                        'monday' => $req["monday"],
                        'tuesday' => $req["tuesday"],
                        'wednesday' => $req["wednesday"],
                        'thursday' => $req["thursday"],
                        'friday' => $req["friday"],
                        'saturday' => $req["saturday"],
                        'sunday' => $req["sunday"],
                        'public_holidays' => $req["public_holidays"]
                    ]
                );
            }

            DB::commit();

            $leave_types = LeaveType::all();

            foreach ($leave_types as $leave_type) {
                $searchCriteria = [
                    "leave_type_id" => $leave_type->id
                ];

                $payroll_setting = PayrollSetting::where($searchCriteria)->first();

                if (!$payroll_setting) {
                    $first_parm_code = CpParmCodes::first();

                    $payroll_setting = PayrollSetting::create(array_merge($searchCriteria, [
                        //get first CpParmCodes
                        'leave_balance_code' => $first_parm_code->OrdinalNo,
                        'leave_taken_code' => $first_parm_code->OrdinalNo,
                    ]));
                }
            }

            DB::commit();

            $payroll_settings = PayrollSetting::all();

            return response()->json([
                'message' => "Leave Types successully edited.",
                'data' => $leave_types,
                'payroll_settings' => $payroll_settings
            ],200); 
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }   

    public function addLeaveGroup(Request $request){
        $rows = $request->rows;

        DB::beginTransaction();


        try {
            foreach ($rows as &$req) {
                LeaveGroup::create([
                    "leave_type_ids" => json_encode($req["leave_type_ids"])
                ]);
            }

            DB::commit();

            $leave_groups_modified = $this->modifyLeaveGroups();
 
            // Return Json Response
            return response()->json([
                'message' => "Leave group successfully added.",
                'data' => $leave_groups_modified
            ],200);
        } catch (\Exception $e) {
            // Return Json Response
            return response()->json([
                'message' => "Something went really wrong!" .$e
            ],500);
        }
    }

    public function getLeaveGroups(){
        $leave_groups_modified = $this->modifyLeaveGroups();

        return response()->json([
            'results' => $leave_groups_modified,
        ], 200);
    }  
    
    public function modifyLeaveGroups(){
        $leave_groups = LeaveGroup::all()->toArray();
        $leave_types = LeaveType::all()->toArray();

        // return $leave_groups;

        $leave_type_ids = [];
        foreach ($leave_types as $leave_type) {
            $leave_type_ids[] = $leave_type["id"];
            // array_push($leave_type_ids, $leave_type["id"]);
        }

        $leave_groups_modified = array_map(function ($group) use ($leave_type_ids) {
            $result = $group;
        
            foreach ($leave_type_ids as $id) {
                $result[$id] = in_array($id, json_decode($group['leave_type_ids'])) ? 1 : 0;
            }
        
            return $result;
        }, $leave_groups);

        return $leave_groups_modified;
    }

    public function editLeaveGroup(Request $request){
        $rows = $request->rows;
        // return response()->json([
        //     'message' => "Leave failed to edit.",
        //     'request' => $rows
        // ],417);
        DB::beginTransaction();
        
        try {
            // & is a reference to the original source so changes are made to original source rather than copy
            foreach ($rows as &$req) {
                // $leave_group = LeaveGroup::findOrFail($req["id"]);
        
                // $leave_group->leave_type_ids = $req["leave_type_ids"];
                // $leave_group->save();

                $leave_group = LeaveGroup::updateOrCreate(
                    ['id' => $req['id']],  // Search criteria
                    ['leave_type_ids' => $req['leave_type_ids']]  // Attributes to update or create
                );
            }

            DB::commit();

            $leave_groups_modified = $this->modifyLeaveGroups();

            return response()->json([
                'message' => "Leave Groups successully edited.",
                'data' => $leave_groups_modified
            ],200); 
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }  

    public function addPublicHoliday(Request $request){
        $rows = $request->rows;

        DB::beginTransaction();

        try {
            foreach ($rows as &$req) {
                PublicHoliday::create([
                    "name" => $req["name"],
                    "date" => explode("T", $req["date"])[0]
                ]);
            }

            DB::commit();
            
            $public_holidays = PublicHoliday::all();
            // Return Json Response
            return response()->json([
                'message' => "Public holiday successfully added.",
                'data' => $public_holidays
            ],200);
        } catch (\Exception $e) {
            // Return Json Response
            return response()->json([
                'message' => "Something went really wrong!" .$e
            ],500);
        }
    }

    public function pullLeaveBalances(Request $request){
        $user= User::where('email', $request->super_admin_email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response([
                'message' => ['These credentials do not match our records.'],
                'success' => false,
            ], 404);
        }

        DB::beginTransaction();

        try{
            $pull_data = [];

            $payroll = $request->payroll;

            $run_date = $request->run_date;
            $rundateFormatted = \DateTime::createFromFormat('d/m/Y', $run_date)->format('Y-m-d 23:59:59');
            // return $rundateFormatted;

            $employees = User::all();
            $available_EmpNos = $employees->pluck('employee_number')->toArray();
            // return $available_EmpNos;
            
            $pulled_leave_taken_data = LeaveTaken::whereIn('employee_number', $available_EmpNos)->where('status', "completed")->where('start_date', '<=', $rundateFormatted)->where('run_date', $run_date)->get();
            $leave_taken_data1 = LeaveTaken::whereIn('employee_number', $available_EmpNos)->where('status', "completed")->where('start_date', '<=', $rundateFormatted)->where('run_date', null)->get();
            $leave_taken_data2 = LeaveTaken::whereIn('employee_number', $available_EmpNos)->where('status', "completed")->where('start_date', '<=', $rundateFormatted)->where('run_date', "")->get();
                                        // ->union($pulled_timesheet_data)
                                        // ->get();
                                        // ->whereBetween('date', [$date_from, $date_to])

            $leave_taken_data = $pulled_leave_taken_data->merge($leave_taken_data1)->merge($leave_taken_data2);
            // return $leave_taken_data;

            $leave_balances = $this->leaveBalancesToJson("all");
            // return $leave_balances;

            foreach ($leave_balances as $leave_balance) {
                $employee_number = $leave_balance["employee_number"];
                // return $employee_number;
                foreach($leave_balance["balances"] as $key => $balance){
                    // return $balance["available"];

                    $leave_taken = $leave_taken_data->where('employee_number', $employee_number)->where('leave_type_id', intval($key));
                    $duration_taken = $leave_taken->pluck('duration')->toArray();
                    $duration_taken = array_sum($duration_taken);
                    // return $duration_taken;
                    //get last item in $leave_taken
                    $leave_taken_last_date = $leave_taken->last();
                    $updated_at = $leave_taken_last_date->updated_at ?? "";
                    // return $updated_at;

                    $dateTime = new DateTime($updated_at);
                    $formattedDate = $dateTime->format('d/m/Y');
                    $formattedTime = $dateTime->format('H:i');
                    
                    $leave_balance_code = intval(PayrollSetting::where('leave_type_id', intval($key))->value('leave_balance_code'));
                    $leave_taken_code = intval(PayrollSetting::where('leave_type_id', intval($key))->value('leave_taken_code'));

                    $dict_to_append = [];
                    $dict_to_append["EmpNo"] = $employee_number;
                    $dict_to_append["OrdinalNo"] = $leave_balance_code;
                    $dict_to_append["Amt"] = floatval($balance["available"]);
                    $dict_to_append["toReplace"] = "true";
                    $dict_to_append["batchNo"] = $formattedDate;
                    $dict_to_append["costCodes"] = "";
                    $dict_to_append["time"] = $formattedTime;
                    $dict_to_append["type"] = "masterfile";

                    $pull_data[] = $dict_to_append;

                    $dict_to_append = [];
                    $dict_to_append["EmpNo"] = $employee_number;
                    $dict_to_append["OrdinalNo"] = $leave_taken_code;
                    $dict_to_append["Amt"] = floatval($duration_taken);
                    $dict_to_append["toReplace"] = "true";
                    $dict_to_append["batchNo"] = $formattedDate;
                    $dict_to_append["costCodes"] = "";
                    $dict_to_append["time"] = $formattedTime;
                    $dict_to_append["type"] = "transaction";
                    
                    $pull_data[] = $dict_to_append;

                    foreach($leave_taken as $leave){
                        $leave->run_date = $run_date;
                        $leave->save();
                    }
                }
            }

            DB::commit();

            return response()->json([
                'message' => "Leave balances successfully pulled.",
                'data' => $pull_data
            ],200);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }

    public function getPublicHolidays(){
        $leave_groups_modified = $this->modifyPublicHolidays();

        return response()->json([
            'results' => $leave_groups_modified,
        ], 200);
    }  

    public function editPublicHoliday(Request $request){
        $rows = $request->rows;
        DB::beginTransaction();
        
        try {
            // & is a reference to the original source so changes are made to original source rather than copy
            foreach ($rows as &$req) {
                $public_holiday = PublicHoliday::findOrFail($req["id"]);

                $public_holiday->name = $req["name"];
                $public_holiday->date = explode("T", $req["date"])[0];
                $public_holiday->save();
            }

            DB::commit();

            $public_holidays = PublicHoliday::all();

            return response()->json([
                'message' => "Public Holidays successully edited.",
                'data' => $public_holidays
            ],200); 
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }       

    public function editAnnualLeaveEntitlement(Request $request){
        $rows = $request->rows;
        DB::beginTransaction();
        
        try {
            // & is a reference to the original source so changes are made to original source rather than copy
            foreach ($rows as &$req) {
                $annual_leave_entitlement = AnnualLeaveEntitlement::where('employee_number', $req["employee_number"])->first();

                $annual_leave_entitlement->days = $req["days"];
                $annual_leave_entitlement->save();
            }

            DB::commit();

            $annual_leave_entitlements = AnnualLeaveEntitlement::where('employee_number', '!=', 1000)->get();

            return response()->json([
                'message' => "Annual Leave Entitlements successully edited.",
                'data' => $annual_leave_entitlements
            ],200); 
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    } 

    public function editPayrollSetting(Request $request){
        $rows = $request->rows;
        DB::beginTransaction();
        
        try {
            // & is a reference to the original source so changes are made to original source rather than copy
            foreach ($rows as &$req) {
                $payroll_setting = PayrollSetting::where('leave_type_id', $req["leave_type_id"])->first();

                $payroll_setting->leave_balance_code = $req["leave_balance_code"];
                $payroll_setting->leave_taken_code = $req["leave_taken_code"];
                $payroll_setting->save();
            }

            DB::commit();

            $payroll_settings = PayrollSetting::all();

            return response()->json([
                'message' => "Payroll Settings successully edited.",
                'data' => $payroll_settings
            ],200); 
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }       

    public function editLinkedAccount(Request $request){
        $rows = $request->rows;
        DB::beginTransaction();
        
        try {
            // & is a reference to the original source so changes are made to original source rather than copy
            foreach ($rows as &$req) {
                //updateOrCreate
                $linked_account = LinkedAccount::updateOrCreate(
                    ['employee_number' => $req["employee_number"]],  // Search criteria
                    [
                        'employee_number' => $req["employee_number"],
                        'linked_accounts' => $req["linked_accounts"]
                    ]  
                );
            }

            DB::commit();

            $linked_accounts = LinkedAccount::where('employee_number', '!=', 1000)->get();

            return response()->json([
                'message' => "Linked Accounts successully edited.",
                'data' => $linked_accounts
            ],200); 
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }       

    public function CRUD(Request $request){
        if ($request->action == "create"){
            $this->addLeaveType($request);
        }
    } 

    public function deleteLeaveTaken(Request $request){
        // return $request;
        try{
            DB::beginTransaction();

            if ($request->to_delete == "LEAVE TAKEN"){
                $leave_taken = LeaveTaken::findOrFail($request->id);
                
                if ($leave_taken->status === 'approved' || $leave_taken->status === 'declined') {
                    
                    return response()->json([
                        'message' => "Leave is already actioned."
                    ], 417);
                } 
    
                LeaveTaken::where('id', $request->id)->delete();
                $request = (object) [
                    "applicant_employee_number" => $request->applicant_employee_number,
                    "id" => $request->id, 
                    "leave_type_id" => $leave_taken->leave_type_id,
                ];
                
            
                //FIX LEAVE BALANCE
                $modify_leave_balances = $this->modifyLeaveBalances($leave_taken, $request, "delete");
                
                if($modify_leave_balances[1] === 417) {
                    DB::rollBack();

                    return response()->json([
                        'message' => $modify_leave_balances[0]
                    ], 417);
                }
                /////

                
                DB::commit();

                $leave_balances_to_json = $this->leaveBalancesToJson($request->applicant_employee_number);
                $all_leave_taken = LeaveTaken::all();
                $employee_leave_taken = $this->getEmployeeLeaveTaken($request->applicant_employee_number);

                return response()->json([
                    'message' => "Leave Successfully deleted",
                    'new_leave_balances' => $leave_balances_to_json,
                    'all_leave_taken' => $all_leave_taken,
                    'employee_leave_taken' => $employee_leave_taken->original["results"],
                ], 200);
            }
        }
        catch(\Exception $e){
            DB::rollBack();

            return response()->json([
                'message' => 'Error: ' . $e
                // 'message' => 'Error: ' . $e->getMessage()
            ], 417);
        }
    }    

    public function deleteLeaveRecord(Request $request){
        try{
            DB::beginTransaction();

            if ($request->to_delete == "LEAVE TAKEN"){
                $leave_taken = LeaveTaken::findOrFail($request->id);
                
                if ($leave_taken->status === 'approved' || $leave_taken->status === 'declined') {
                    
                    return response()->json([
                        'message' => "Leave is already actioned."
                    ], 417);
                } 
    
                LeaveTaken::where('id', $request->id)->delete();
                $request = (object) [
                    "id" => $request->id, 
                    "leave_type_id" => $leave_taken->leave_type_id,
                ];
                
            
                //FIX LEAVE BALANCE
                $modify_leave_balances = $this->modifyLeaveBalances($leave_taken, $request, "delete");
                
                if($modify_leave_balances[1] === 417) {
                    DB::rollBack();

                    return response()->json([
                        'message' => $modify_leave_balances[0]
                    ], 417);
                }
                /////

                
                DB::commit();

                $leave_balances_to_json = $this->leaveBalancesToJson($leave_taken->employee_number);
                $all_leave_taken = LeaveTaken::all();
                $employee_leave_taken = $this->getEmployeeLeaveTaken($leave_taken->employee_number);

                return response()->json([
                    'message' => "Leave Successfully deleted",
                    'new_leave_balances' => $leave_balances_to_json,
                    'all_leave_taken' => $all_leave_taken,
                    'employee_leave_taken' => $employee_leave_taken->original["results"],
                ], 200);
            } else if ($request->to_delete == "LEAVE TYPE"){
                $leave_type = LeaveType::findOrFail($request->id);
                LeaveType::where('id', $request->id)->delete();

                DB::commit();

                return response()->json([
                    'message' => "Leave Type Successfully deleted",
                    'payroll_settings' => PayrollSetting::all(),
                ], 200);
            } else if ($request->to_delete == "LEAVE GROUP"){
                $leave_Group = LeaveGroup::findOrFail($request->id);
                LeaveGroup::where('id', $request->id)->delete();

                DB::commit();

                return response()->json([
                    'message' => "Leave Group Successfully deleted",
                ], 200);
            } else if ($request->to_delete == "PUBLIC HOLIDAY"){
                $leave_Group = PublicHoliday::findOrFail($request->id);
                PublicHoliday::where('id', $request->id)->delete();

                DB::commit();

                return response()->json([
                    'message' => "Public Holiday Successfully deleted",
                ], 200);
            } else if ($request->to_delete == "LINKED ACCOUNTS"){
                $linked_account = LinkedAccount::findOrFail($request->id);
                LinkedAccount::where('id', $request->id)->delete();

                DB::commit();

                return response()->json([
                    'message' => "Linked Account Successfully deleted",
                ], 200);
            }
        }
        catch(\Exception $e){
            DB::rollBack();

            return response()->json([
                'message' => 'Error: ' . $e->getMessage()
            ], 417);
        }
    }    
    
    public function setLeaveBalances(Request $request){
        // return $request;
        $rows = $request->rows;
        DB::beginTransaction();

        try {
            foreach ($rows as &$req) {
                $leave_balances = LeaveBalance::where('employee_number', $req["employee_number"])->first();

                if ($leave_balances) {
                    $decoded_json = json_decode($leave_balances["balances"], true);
                    $json_keys = array_keys($req["balances"]);
                    foreach ($json_keys as $key){
                        if ($req["balances"][$key]["available"] !== null){
                            $req["balances"][$key]["pending"] = (array_key_exists($key, $decoded_json)) ? $decoded_json[$key]["pending"] : 0;
                        } else {
                            unset($req["balances"][$key]);
                        }
                    }
                    $leave_balances->balances = json_encode($req["balances"]);
                    $leave_balances->save();
                }
                else {
                    $json_keys = array_keys($req["balances"]);
                    foreach ($json_keys as $key){
                        if ($req["balances"][$key]["available"] !== null){
                            $req["balances"][$key]["pending"] = 0;
                        } else {
                            unset($req["balances"][$key]);
                        }
                    }

                    LeaveBalance::create([
                        "employee_number" => $req["employee_number"],
                        "balances" => json_encode($req["balances"])
                    ]);
                }
            }

            DB::commit();

            $leave_balances_to_json = $this->leaveBalancesToJson("all");

            return response()->json([
                'message' => "Leave balances successfully added.",
                'data' => $leave_balances_to_json
            ],200);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }

    public function getUserLeaveBalances($id){
        $leave_balances = LeaveBalance::where('employee_number', $id)->firstOrFail();

        return response()->json([
            'results' => $leave_balances
        ], 200);
    }

    public function getLeaveBalances(){
        $leave_balances_to_json = $this->leaveBalancesToJson("all");

        return response()->json([
            'results' => $leave_balances_to_json,
        ], 200);
    }  

    public function leaveBalancesToJson($employee_number){
        if($employee_number !== "all"){
            $linked_accounts = LinkedAccount::where('employee_number', $employee_number)->first();
            if ($linked_accounts){
                $linked_accounts = json_decode($linked_accounts->linked_accounts, true);
                //add $employee_number to array
                array_push($linked_accounts, $employee_number);
                $leave_balances = LeaveBalance::whereIn('employee_number', $linked_accounts)->get()->toArray();
            } else {
                $leave_balances = LeaveBalance::where("employee_number", $employee_number)->get()->toArray();
            }
        } else {
            $leave_balances = LeaveBalance::where('employee_number', '!=', 1000)->get()->toArray();
        }

        $leave_types = LeaveType::all()->toArray();

        // return $leave_balances;

        $leave_type_ids = [];
        foreach ($leave_types as $leave_type) {
            $leave_type_ids[] = $leave_type["id"];
            // array_push($leave_type_ids, $leave_type["id"]);
        }

        $leave_balances_to_json = array_map(function ($balance) use ($leave_type_ids) {
            $result = $balance;
            $result["balances"] = [];
        
            foreach ($leave_type_ids as $id) {
                $result["balances"][$id] = array_key_exists($id, json_decode($balance['balances'], true)) ? json_decode($balance['balances'], true)[$id] : ["pending" => "", "available" => ""];
            }
        
            return $result;
        }, $leave_balances);

        return $leave_balances_to_json;
    } 
    
    public function modifyLeaveBalances($leave_taken, $request, $modify_type){
        $leave_balances = LeaveBalance::where('employee_number', $leave_taken->employee_number)->first();
            
        if ($leave_balances) {
            $balances = json_decode($leave_balances->balances, true);
            
            //add back the leave
            if ($modify_type === "edit" || $modify_type === "delete" || $modify_type === "declined") {
                $balances[$request->leave_type_id]['available'] += $leave_taken->duration; //number_format($value, 2, '.', '');$leave_taken->duration
                $balances[$request->leave_type_id]['pending'] -= $leave_taken->duration;
            }

            //subtract the new leave
            if ($modify_type === "edit" || $modify_type === "create" || $modify_type === "pending") {
                $balances[$request->leave_type_id]['available'] -= $request->duration;
                $balances[$request->leave_type_id]['pending'] += $request->duration;
                
                //if leave policy doesn't allow less than 0. Fail safe from check on the FE
                if ($balances[$request->leave_type_id]['available'] < 0) {
                    return ["Leave Balance Not Sufficient.", 417];

                    // return response()->json([
                    //     'message' => "Leave Balance Not Sufficient."
                    // ], 417);
                }
            }

            //approve leave //subtract from to pending
            if ($modify_type === "approved") {
                $balances[$request->leave_type_id]['pending'] -= $leave_taken->duration;
            }

            $balances = json_encode($balances);
            
            $leave_balances->balances = $balances;
            $leave_balances->save();

            return ["Successfull Balance Modification.", 200];
        } else {
            return ["Leave Balances Not Found.", 417];
            
            // return response()->json([
            //     'message' => "Leave Balances Not Found."
            // ], 417);
        }    
    }        

    public function getLeaveTaken($id){
        $leave_taken = LeaveTaken::where('employee_number', $id)->get();

        return response()->json([
            'results' => $leave_taken
        ], 200);
    }

    // public function getApproveeLeaveTaken($id){ WHEN LEAVE APPROVERS AND TYPE ARE IN GROUP
    //     $id = intval($id);
    //     $leave_groups = LeaveGroup::all();

    //     //get leave groups where the id exists in the array located in the leave_approvers column
    //     $leave_groups_with_approver = $leave_groups->filter(function ($group) use ($id) {
    //         return in_array($id, json_decode($group['leave_approvers']));
    //     });

    //     $approvees = User::whereIn('leave_group_id', $leave_groups_with_approver->pluck('id'))->get();

    //     $approvees_leave_taken = LeaveTaken::whereIn('employee_number', $approvees->pluck('employee_number'))->get();

    //     //logic to only see leave one is supposed to see
    //     $approvees_leave_taken = $approvees_leave_taken->filter(function ($leave_taken) use ($id) {
    //         $employee = User::where('employee_number', $leave_taken->employee_number)->first();
    //         $leave_group = LeaveGroup::where('id', $employee->leave_group_id)->get()[0];
    //         $approvers = json_decode($leave_group->leave_approvers);
    //         //check if length of array approvers is greater than 1 and type is heirachichal and leave is not approved
    //         if((count($approvers) > 1) && ($leave_group->leave_approvers_type === "hierachichal") && ($leave_taken->status !== 'approved')) {
    //             if ($leave_taken->status === "pending"){
    //                 //if first to approve
    //                 if (empty($leave_taken->actioned_by)){
    //                     if($approvers[0] === $id){
    //                         return true;
    //                     } else {
    //                         return false;
    //                     }
    //                 } else { //if next in line to approve or in actioned by
    //                     $actioned_by_list = json_decode($leave_taken->actioned_by);
    //                     $latest_approver_index = array_search(end($actioned_by_list), $approvers);
    //                     $next_approver = $approvers[$latest_approver_index + 1];
    
    //                     if($next_approver === $id || in_array($id, json_decode($leave_taken->actioned_by))){
    //                         return true;
    //                     } else {
    //                         return false;
    //                     }
    //                 }

    //             } else if ($leave_taken->status === "declined"){
    //                 if (in_array($id, json_decode($leave_taken->actioned_by))){
    //                     return true;
    //                 } else {
    //                     return false;
    //                 }
    //             } else {
    //                 return false;
    //             }
    //         } else {
    //             return true;
    //         }
    //     });


    //     return response()->json([
    //         'results' => $approvees_leave_taken
    //     ], 200);
    // }

    public function getApproveeLeaveTaken($id){
        try{
            $id = intval($id);
            $leave_takens = LeaveTaken::all();
            $leave_types = LeaveType::all();

            //logic to only see leave one is supposed to see
            $approvees_leave_taken = $leave_takens->filter(function ($leave_taken) use ($leave_types, $id) {
                $leave_type = $leave_types->filter(function ($leave_type) use ($leave_taken) {
                    return ($leave_type["id"] == $leave_taken->leave_type_id);
                })->first();

                // $leave_type = $leave_type[0];

                $approvers = json_decode($leave_type->leave_approvers);

                //check if approver exists in leave_type approvers
                if (in_array($id, $approvers)){
                    //check if length of array approvers is greater than 1 and type is heirachichal and leave is not approved
                    if((count($approvers) > 1) && ($leave_type->leave_approvers_type === "hierachichal") && ($leave_taken->status !== 'approved')) {
                        if ($leave_taken->status === "pending"){
                            //if first to approve
                            if (empty($leave_taken->actioned_by)){
                                if($approvers[0] === $id){
                                    return true;
                                } else {
                                    return false;
                                }
                            } else { //if next in line to approve or in actioned by
                                $actioned_by_list = json_decode($leave_taken->actioned_by);
                                $latest_approver_index = array_search(end($actioned_by_list), $approvers);
                                $next_approver = $approvers[$latest_approver_index + 1];
            
                                if($next_approver === $id || in_array($id, json_decode($leave_taken->actioned_by))){
                                    return true;
                                } else {
                                    return false;
                                }
                            }

                        } else if ($leave_taken->status === "declined"){
                            if (in_array($id, json_decode($leave_taken->actioned_by))){
                                return true;
                            } else {
                                return false;
                            }
                        } else {
                            return false;
                        }
                    } else {
                        return true;
                    }
                } else {
                    return false;
                }
            });

            return response()->json([
                'results' => $approvees_leave_taken
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }

    public function getEmployeeLeaveTaken($employee_number){
        $linked_accounts = LinkedAccount::where('employee_number', $employee_number)->first();
        if ($linked_accounts){
            $linked_accounts = json_decode($linked_accounts->linked_accounts, true);
            //add $employee_number to array
            array_push($linked_accounts, $employee_number);
            $employee_leave_taken = LeaveTaken::whereIn('employee_number', $linked_accounts)->get();
        } else {
            $employee_leave_taken = LeaveTaken::where("employee_number", $employee_number)->get();
        }

        foreach($employee_leave_taken as $leave_taken){
            $leave_type = LeaveType::where('id', $leave_taken->leave_type_id)->first();
            $leave_taken->title = $leave_type->name;
        }

        return response()->json([
            'results' => $employee_leave_taken
        ], 200);
    }

    public function applyLeaveTaken(Request $request){
        try {
            // Start a database transaction
            DB::beginTransaction();

            // $employee = User::findOrFail($request["employee_number"]);
            $employee = User::where('employee_number', $request["employee_number"])->first();

            //find user or fail with customized message
            if ($employee == null) {
                return response()->json([
                    'message' => "Employee not found."
                ], 417);
            }

            //check if leave_group is null
            $leave_group = LeaveGroup::find($employee->leave_group_id);
            if ($leave_group == null) {
                return response()->json([
                    'message' => "Employee Leave Group not found."
                ], 417);
            } 

            //check if leave_approvers is null
            $leave_type = LeaveType::find($request->leave_type_id);
            if ($leave_type->leave_approvers == null) {
                return response()->json([
                    'message' => "Employee Leave Type has no leave approvers."
                ], 417);
            }

            $leave_approvers = json_decode($leave_type->leave_approvers, true);
            $first_leave_approver = User::where('employee_number', $leave_approvers[0])->first();

            //check if leave_approver exists
            if ($first_leave_approver == null) {
                return response()->json([
                    'message' => "Approver not found."
                ], 417);
            }

            //logic to avoid overlapping of leave
            $existing_leave_taken = LeaveTaken::where('employee_number', $request->employee_number)
                                    ->where('start_date', '<=', $request->end_date)
                                    ->where('end_date', '>=', $request->start_date)
                                    ->where('status', '<>', "declined")
                                    ->get();
            // return $existing_leave_taken;

            if ($existing_leave_taken->count() > 0) {
                // Roll back the transaction if the leave overlaps
                DB::rollBack();

                return response()->json([
                    'message' => "Leave overlaps with existing leave.",
                    'exisiting' => $existing_leave_taken
                ], 417);
            }

            
            //now creating the leave
            $leave_taken = LeaveTaken::create([
                "employee_number" => $request->employee_number,
                "leave_type_id" => $request->leave_type_id,
                "start_date" => $request->start_date,
                "end_date" => $request->end_date,
                "duration" => $request->duration,
                "application_reason" => $request->application_reason,
                "status" => $request->status
            ]);

            //SAVE FILE TO STORAGE
            // Validate the request to ensure a file was uploaded
            // $request->validate([
            //     'attachment' => 'required|file|mimes:pdf,doc,docx|max:2048',
            // ]);

            if ($request->hasFile('attachment')) {
                $file = $request->file('attachment');

                // Perform further processing with the file
                $fileName = $file->getClientOriginalName();
                $fileExtension = $file->getClientOriginalExtension();
                // ... (Do whatever you need with the file)

                // Generate a new filename
                $newFileName = $leave_taken->id .'_'. $leave_taken->employee_number .".". $fileExtension;

                // Move the uploaded file to a storage location
                $file->storeAs('attachments', $newFileName, 'public'); // Adjust the storage path as needed
                $leave_taken->attachment = asset('storage/attachments/' . $newFileName);
                $leave_taken->save();
            }


            //FIX LEAVE BALANCE
            $modify_leave_balances = $this->modifyLeaveBalances($leave_taken, $request, "create");

            if($modify_leave_balances[1] === 417) {
                DB::rollBack();

                return response()->json([
                    'message' => $modify_leave_balances[0]
                ], 417);
            }
            /////


            // Commit the transaction if all operations are successful
            DB::commit();

            $leave_balances_to_json = $this->leaveBalancesToJson($request->applicant_employee_number);

            $employee_leave_taken = $this->getEmployeeLeaveTaken($request->applicant_employee_number);

            $email_data = [
                    "title" => "Leave Approval Request",
                    "email" => $first_leave_approver->email,
                    "blade" => 'emails.leave_approval_request',
                    "displayableActionUrl" => env("FRONTEND_URL"),
                    "actionUrl" => env("FRONTEND_URL"),
                    "color" => "blue",
                    "greeting" => "Good day",
                    "introLines" => ["You have a new leave request from ".$employee->name." ".$employee->surname],
                    "actionText" => "Login",
                    "outroLines" => ["Please review the request and action."],
                    "salutation" => "Regards,\n".env("APP_NAME") 
            ];
            $email_data = (object) $email_data;
            $email_response = $this->sendLeaveRequestEmail($email_data, "leave approval request");
 
            return response()->json([
                'message' => "Leave successully applied.",
                'leave_id' => $leave_taken->id,
                'new_leave_balances' => $leave_balances_to_json,
                'employee_leave_taken' => $employee_leave_taken->original["results"],
                'email_response' => $email_response
            ],200);
        } catch (\Exception $e) {
            // Handle the exception and roll back the transaction
            DB::rollBack();

            return response()->json([
                'message' => 'Error: ' . $e
                // 'message' => 'Error: ' . $e->getMessage()
            ],500);
        }
    }

    public function editLeaveTaken(Request $request){
        // return response()->json([
        //     'message' => "Leave failed to edit."
        // ],417);
        
        try {
            $todays_date = date('Y-m-d');

            DB::beginTransaction();

            $leave_taken = LeaveTaken::findOrFail($request->id);
            
            if ($leave_taken){
                //check if leave has been approved or completed to avoid editing
                //if approved and day has not passed, can edit and change balance but if day has passed, can't edit
                // if ($leave_taken->status === 'approved' || ($leave_taken->end_date < $todays_date && $leave_taken->status === 'approved')) {
                // if ($leave_taken->status === 'approved' && $leave_taken->end_date < $todays_date) {
                // if ($leave_taken->status === 'approved' && $request->end_date < $todays_date) {
                if ($leave_taken->status === 'approved' || $leave_taken->status === 'declined') {
                    DB::rollBack();

                    return response()->json([
                    'message' => "Leave is already actioned."
                    ], 417);
                } 

                //logic to avoid overlapping of leave
                $existing_leave_taken = LeaveTaken::where('employee_number', $leave_taken->employee_number)
                                        ->where('id', '<>', $request->id)
                                        ->where('start_date', '<=', $request->end_date)
                                        ->where('end_date', '>=', $request->start_date)
                                        ->where('status', '<>', "declined")
                                        ->get();
                // return $existing_leave_taken;

                if ($existing_leave_taken->count() > 0) {
                    // Roll back the transaction if the leave overlaps
                    DB::rollBack();

                    return response()->json([
                        'message' => "Leave overlaps with existing leave.",
                        'exisiting' => $existing_leave_taken
                    ], 417);
                }

                //if leave is not all day or is manualDuration, should not be able to move to outside working hours
                if(count(explode('T', $request->start_date)) > 1){ //this means it's manual duration
                    $leave_type_start_time = LeaveType::where('id', $leave_taken->leave_type_id)->first()->start_time;
                    $leave_type_end_time = LeaveType::where('id', $leave_taken->leave_type_id)->first()->end_time;
                    
                    $new_leave_start_time = explode('T', $request->start_date)[1];
                    $new_leave_end_time = explode('T', $request->end_date)[1];

                    $leave_type_start_time = Carbon::createFromFormat('H:i', $leave_type_start_time);
                    $leave_type_end_time = Carbon::createFromFormat('H:i', $leave_type_end_time);
                    $new_leave_start_time = Carbon::createFromFormat('H:i:s', $new_leave_start_time);
                    $new_leave_end_time = Carbon::createFromFormat('H:i:s', $new_leave_end_time);

                    // return [$leave_type_start_time, $leave_type_end_time, $new_leave_start_time, $new_leave_end_time];

                    //check if leave is outside working hours
                    if($new_leave_start_time->lessThan($leave_type_start_time) || $new_leave_end_time->greaterThan($leave_type_end_time)){
                        DB::rollBack();

                        return response()->json([
                            'message' => "Leave is outside working hours."
                        ], 417);
                    }
                }



                // //REFUSE TO ADJUST END DATE BELOW TODAY'S DATE
                // if ($request->end_date < $todays_date) {
                //     DB::rollBack();

                //     return response()->json([
                //     'message' => "End date cannot be before today's date."
                //     ], 417);
                // }


                //FIX LEAVE BALANCE
                $modify_leave_balances = $this->modifyLeaveBalances($leave_taken, $request, "edit");

                if($modify_leave_balances[1] === 417) {
                    DB::rollBack();

                    return response()->json([
                        'message' => $modify_leave_balances[0]
                    ], 417);
                }
                /////

                $leave_taken->leave_type_id = $request->leave_type_id;
                $leave_taken->start_date = $request->start_date;
                $leave_taken->end_date = $request->end_date;
                $leave_taken->duration = $request->duration;
                $leave_taken->application_reason = $request->application_reason;
                $leave_taken->attachment = $request->attachment;
                $leave_taken->status = $request->status;
                $leave_taken->save();
    
                // Commit the transaction if all operations are successful
                DB::commit();

                $leave_balances_to_json = $this->leaveBalancesToJson($request->applicant_employee_number);
                $employee_leave_taken = $this->getEmployeeLeaveTaken($request->applicant_employee_number);

                return response()->json([
                    'message' => "Leave successully edited.",
                    'new_leave_balances' => $leave_balances_to_json,
                    'employee_leave_taken' => $employee_leave_taken->original["results"],
                ],200);
            } else {
                DB::rollBack();
                
                return response()->json([
                    'message' => "Leave not found."
                ],417);
            }
 
        } catch (\Exception $e) {
            // Return Json Response
            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }

    public function getAccrualRuns(){
        $accrual_runs = AccrualRun::all();

        return response()->json([
            'success' => true,
            'results' => $accrual_runs,
        ], 200);

    }

    public function accrualRun(Request $request){        
        try {
            DB::beginTransaction();
           
            //get the least value in the accrual_type column in LeaveType and get only the value in the accrual_type column
            $leave_types = LeaveType::all();
            $least_value = LeaveType::orderBy('accrual_type', 'asc')->first();
            $least_value = $least_value->accrual_type;
            $total_runs_per_month = 1/$least_value;

            if($request->action === "run"){               
                $accrual_run = AccrualRun::orderBy('id', 'desc')->first();
                if($accrual_run){
                    //get the latest accrual run done
                    //check if month runs are all done
                    if(($accrual_run->total_runs / $accrual_run->run_number) === 1){
                        $month = $accrual_run->month + 1;
                        $run_number = 1;
                    } else {
                        $month = $accrual_run->month;
                        $run_number = $accrual_run->run_number + 1;
                    }
                
                } else {
                    $month = 1;
                    $run_number = 1;
                }

                AccrualRun::create([
                    'month' => $month,
                    'run_number' => $run_number,
                    'total_runs' => $total_runs_per_month,
                ]);

                //change all leaveTaken to completed if end_date is less than today's date
                $todays_date = date('Y-m-d');
                $leave_taken = LeaveTaken::where('end_date', '<', $todays_date)->get();

                foreach($leave_taken as $lt){
                    if($lt->status === "approved"){
                        $lt->status = "completed";
                        $lt->save();
                    }
                }

                //change leave balances for all users without stop_accrual === 1, by adding the accrual days from the leave_types to each balances
                $leave_balances = LeaveBalance::all();

                foreach($leave_balances as $lb){
                    $employee_number = $lb->employee_number;
                    $stop_accrual = EmployeeInformation::where('employee_number', $employee_number)->first()->stop_accrual;
                    $annual_leave_entitlement = AnnualLeaveEntitlement::where('employee_number', $employee_number)->first()->days;
                    if($stop_accrual != 1){
                        $balances = json_decode($lb->balances, true);
                        foreach($leave_types as $lt){
                            
                            //check if array key exists 
                            if(array_key_exists(strval($lt->id), $balances)){
                                //check if accrual type is less than monthly
                                if(($total_runs_per_month > 1) && ($run_number != $total_runs_per_month)){
                                    if($run_number/$total_runs_per_month === $lt->accrual_type || $lt->accrual_type == 0.25){
                                        $annual_leave_entitlement = $annual_leave_entitlement/(12/$lt->accrual_type);
                                        $accrual_days = $lt->name === "ANNUAL" ? $annual_leave_entitlement : $lt->accrual_days;
                                        $balances[strval($lt->id)]["available"] = $balances[strval($lt->id)]["available"] + $accrual_days;
                                    }
                                } else {
                                    $hasNoRemainder = fmod($month, $lt->accrual_type) == 0;
                                    if($hasNoRemainder){
                                        $annual_leave_entitlement = $annual_leave_entitlement/(12/$lt->accrual_type);
                                        $accrual_days = $lt->name === "ANNUAL" ? $annual_leave_entitlement : $lt->accrual_days;
                                        $balances[strval($lt->id)]["available"] = $balances[strval($lt->id)]["available"] + $accrual_days;
                                    }
                                }
                            }
                        }
                        
                        $lb->balances = json_encode($balances);
                        $lb->save();
                    }
                }
            } else { //action === "undo"
                //get the last row in AccrualRun and delete
                $accrual_run = AccrualRun::orderBy('id', 'desc')->first();
                if($accrual_run){
                    $month = $accrual_run->month;
                    $run_number = $accrual_run->run_number;
                    
                    $accrual_run->delete();

                    //change all leaveTaken back to approved if status === completed
                    $todays_date = date('Y-m-d');
                    $leave_taken = LeaveTaken::where('end_date', '<', $todays_date)->get();

                    foreach($leave_taken as $lt){
                        if($lt->status === "completed"){
                            $lt->status = "approved";
                            $lt->save();
                        }
                    }

                    //undo leave balances accrual
                    $leave_balances = LeaveBalance::all();

                    foreach($leave_balances as $lb){
                        $employee_number = $lb->employee_number;
                        $stop_accrual = EmployeeInformation::where('employee_number', $employee_number)->first()->stop_accrual;
                        $annual_leave_entitlement = AnnualLeaveEntitlement::where('employee_number', $employee_number)->first()->days;
                        if($stop_accrual != 1){

                            $balances = json_decode($lb->balances, true);
                            foreach($leave_types as $lt){
                                
                                //check if array key exists 
                                if(array_key_exists(strval($lt->id), $balances)){
                                    //check if accrual type is less than monthly
                                    if(($total_runs_per_month > 1) && ($run_number != $total_runs_per_month)){
                                        if($run_number/$total_runs_per_month === $lt->accrual_type || $lt->accrual_type == 0.25){
                                            $annual_leave_entitlement = $annual_leave_entitlement/(12/$lt->accrual_type);
                                            $accrual_days = $lt->name === "ANNUAL" ? $annual_leave_entitlement : $lt->accrual_days;
                                            $balances[strval($lt->id)]["available"] = $balances[strval($lt->id)]["available"] - $accrual_days;
                                        }
                                    } else {
                                        $hasNoRemainder = fmod($month, $lt->accrual_type) == 0;
                                        if($hasNoRemainder){
                                            $annual_leave_entitlement = $annual_leave_entitlement/(12/$lt->accrual_type);
                                            $accrual_days = $lt->name === "ANNUAL" ? $annual_leave_entitlement : $lt->accrual_days;
                                            $balances[strval($lt->id)]["available"] = $balances[strval($lt->id)]["available"] - $accrual_days;
                                        }
                                    }
                                }
                            }
                            
                            $lb->balances = json_encode($balances);
                            $lb->save();
                        }
                            
                    }

                }
                
            }
            
            DB::commit();  
            return response()->json([
                'message' => "Successful",
                'accrual_runs' => AccrualRun::all(),
                'leave_balances' => $this->leaveBalancesToJson("all"),
            ], 200);
            
        } catch (\Exception $e) {
            DB::rollBack();
            // Return Json Response
            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }

    public function sendLeaveRequestEmail($request, $type){
        // return response()->json([
        //     'message' => "Leave failed to edit."
        // ],417);
        
        try {
            $data = [
                "level" => "",
                "displayableActionUrl" => $request->displayableActionUrl,
                "actionUrl" => $request->actionUrl,
                "color" => $request->color,
                "greeting" => $request->greeting,
                "introLines" => $request->introLines,
                "actionText" => $request->actionText,
                "outroLines" => $request->outroLines,
                "salutation" => $request->salutation
            ];

            if($type === "leave approval request"){
                Mail::to($request->email)->send(new LeaveApprovalRequestNotification($data));
            } else if ($type === "leave actioned") {
                Mail::to($request->email)->send(new LeaveActionedNotification($data));
            }

            return "Email Sent";
            // Mail::send($request->blade, $data, function ($message) use ($request) {
            //     $message->to($request->email)->subject($request->title);
            // });

 
        } catch (\Exception $e) {
            // Return Json Response
            return response()->json([
                'message' => "Something went really wrong! .$e"
            ],500);
        }
    }
}
